<?php
/**
* 2011-2019 Promokit
*
* @package   pk_amp
* @version   1.0
* @author    https://promokit.eu
* @copyright Copyright Ⓒ 2019 promokit.eu <@email:support@promokit.eu>
* @license   GNU General Public License version 2
*/

require_once(_PS_MODULE_DIR_.'pk_amp/classes/Amp.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutProcess.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutSession.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutPersonalInformationStep.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutAddressesStep.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutDeliveryStep.php');
require_once (_PS_CLASS_DIR_.'checkout/CheckoutPaymentStep.php');
require_once(_PS_FRONT_CONTROLLER_DIR_.'OrderController.php');
use PrestaShop\PrestaShop\Core\Foundation\Templating\RenderableProxy;
use PrestaShop\PrestaShop\Adapter\Product\PriceFormatter;

class Pk_AmpCheckoutModuleFrontController extends ModuleFrontController
{
    public $ssl = true;
    public $name = 'checkout';
    protected $checkoutProcess;
    protected $cartChecksum;

    public function init()
    {
        parent::init();
        $this->cartChecksum = new CartChecksum(new AddressChecksum());
    }
    
    public function initContent()
    {
        $ampClass = new Amp();
        $ampClass->injectCSS(array('checkout'));

        if (Configuration::isCatalogMode()) {
            Tools::redirect('index.php');
        }

        $this->restorePersistedData($this->checkoutProcess);
        $this->checkoutProcess->handleRequest(
            Tools::getAllValues()
        );

        $presentedCart = $this->cart_presenter->present($this->context->cart);

        if (count($presentedCart['products']) <= 0 || $presentedCart['minimalPurchaseRequired']) {
            Tools::redirect('index.php?controller=cart');
        }

        $this->checkoutProcess->setNextStepReachable();
        $this->checkoutProcess->markCurrentStep();
        $this->checkoutProcess->invalidateAllStepsAfterCurrent();        

        $this->saveDataToPersist($this->checkoutProcess);

        if (!$this->checkoutProcess->hasErrors()) {
            if ($_SERVER['REQUEST_METHOD'] !== 'GET' && !$this->ajax) {
                return $this->redirectWithNotifications(
                    $this->checkoutProcess->getCheckoutSession()->getCheckoutURL()
                );
            }
        }

        $cp = new RenderableProxy($this->checkoutProcess);
        
        $this->context->smarty->assign(array(
            'checkout_process' => $cp,
            'cart' => $presentedCart,
            'display_transaction_updated_info' => Tools::getIsset('updatedTransaction'),
            'canonical' => $this->context->link->getPageLink($this->name),
            'amp' => $ampClass->getThemeVars(array('name' => $this->name))
        ));

        parent::initContent();
        $this->setTemplate('mobile/checkout/'.$this->name);
    }

    public function postProcess()
    {
        parent::postProcess();
        $this->bootstrap();

        if (Tools::isSubmit('submitReorder') && $id_order = (int) Tools::getValue('id_order')) {
            $oldCart = new Cart(Order::getCartIdStatic($id_order, $this->context->customer->id));
            $duplication = $oldCart->duplicate();
            if (!$duplication || !Validate::isLoadedObject($duplication['cart'])) {
                $this->errors[] = $this->trans('Sorry. We cannot renew your order.', array(), 'Shop.Notifications.Error');
            } elseif (!$duplication['success']) {
                $this->errors[] = $this->trans(
                    'Some items are no longer available, and we are unable to renew your order.', array(), 'Shop.Notifications.Error'
                );
            } else {
                $this->context->cookie->id_cart = $duplication['cart']->id;
                $context = $this->context;
                $context->cart = $duplication['cart'];
                CartRule::autoAddToCart($context);
                $this->context->cookie->write();
                Tools::redirect('index.php?controller=order');
            }
        }

        if (Tools::getValue('confirm-addresses') == 1) {

            $addressForm = $this->makeAddressForm();

            if (Tools::getIsset('id_address') && ($id_address = (int)Tools::getValue('id_address'))) {
                $addressForm->loadAddressById($id_address);
            }

            if (Tools::getIsset('id_country')) {
                $addressForm->fillWith(array('id_country' => Tools::getValue('id_country')));
            }

            $stepTemplateParameters = array();
            foreach ($this->checkoutProcess->getSteps() as $step) {
                if ($step instanceof CheckoutAddressesStep) {
                    $stepTemplateParameters = $step->getTemplateParameters();
                }
            }

            $templateParams = array_merge(
                $addressForm->getTemplateVariables(),
                $stepTemplateParameters,
                array('type' => 'delivery')
            );
            ob_end_clean();

            $link = $this->context->link->getModuleLink('pk_amp', 'checkout', array(), true, $this->context->language->id, $this->context->shop->id);

            header("access-control-allow-credentials:true");
            header("access-control-allow-headers:Content-Type, Content-Length, Accept-Encoding, X-CSRF-Token");
            header("access-control-allow-methods:GET,POST");
            header("amp-access-control-allow-source-origin: https://".$_SERVER['HTTP_HOST']);
            header("access-control-allow-origin: https://".$_SERVER['HTTP_HOST']);
            header("AMP-Redirect-To:".$link);

        }

        if (Tools::getValue('confirmDeliveryOption') == 1) {

            $cart = $this->cart_presenter->present(
                $this->context->cart
            );
            
            $link = $this->context->link->getModuleLink('pk_amp', 'checkout', array(), true, $this->context->language->id, $this->context->shop->id);

            header("access-control-allow-credentials:true");
            header("access-control-allow-headers:Content-Type, Content-Length, Accept-Encoding, X-CSRF-Token");
            header("access-control-allow-methods:GET,POST");
            header("amp-access-control-allow-source-origin: https://".$_SERVER['HTTP_HOST']);
            header("access-control-allow-origin: https://".$_SERVER['HTTP_HOST']);
            header("AMP-Redirect-To:".$link);

        }

    }

    protected function getCheckoutSession()
    {
        $deliveryOptionsFinder = new DeliveryOptionsFinder(
            $this->context,
            $this->getTranslator(),
            $this->objectPresenter,
            new PriceFormatter()
        );

        $session = new CheckoutSession(
            $this->context,
            $deliveryOptionsFinder
        );

        return $session;
    }

    public function bootstrap()
    {
        $translator = $this->getTranslator();

        $session = $this->getCheckoutSession();

        $this->checkoutProcess = new CheckoutProcess(
            $this->context,
            $session
        );

        $this->checkoutProcess
            ->addStep(new CheckoutPersonalInformationStep(
                $this->context,
                $translator,
                $this->makeLoginForm(),
                $this->makeCustomerForm()
            ))
            ->addStep(new CheckoutAddressesStep(
                $this->context,
                $translator,
                $this->makeAddressForm()
            ));

        if (!$this->context->cart->isVirtualCart()) {
            $checkoutDeliveryStep = new CheckoutDeliveryStep(
                $this->context,
                $translator
            );

            $checkoutDeliveryStep
                ->setRecyclablePackAllowed((bool) Configuration::get('PS_RECYCLABLE_PACK'))
                ->setGiftAllowed((bool) Configuration::get('PS_GIFT_WRAPPING'))
                ->setIncludeTaxes(
                    !Product::getTaxCalculationMethod((int) $this->context->cart->id_customer)
                    && (int) Configuration::get('PS_TAX')
                )
                ->setDisplayTaxesLabel((Configuration::get('PS_TAX') && !Configuration::get('AEUC_LABEL_TAX_INC_EXC')))
                ->setGiftCost(
                    $this->context->cart->getGiftWrappingPrice(
                        $checkoutDeliveryStep->getIncludeTaxes()
                    )
                );

            $this->checkoutProcess->addStep($checkoutDeliveryStep);
        }

        $this->checkoutProcess
            ->addStep(new CheckoutPaymentStep(
                $this->context,
                $translator,
                new PaymentOptionsFinder(),
                new ConditionsToApproveFinder(
                    $this->context,
                    $translator
                )
            ))
        ;
    }

    private function saveDataToPersist(CheckoutProcess $process)
    {
        $data = $process->getDataToPersist();
        $data['checksum'] = $this->cartChecksum->generateChecksum($this->context->cart);

        Db::getInstance()->execute(
            'UPDATE '._DB_PREFIX_.'cart SET checkout_session_data = "'.pSQL(json_encode($data)).'"
                WHERE id_cart = '.(int) $this->context->cart->id
        );
    }

    private function restorePersistedData(CheckoutProcess $process)
    {
        $rawData = Db::getInstance()->getValue(
            'SELECT checkout_session_data FROM '._DB_PREFIX_.'cart WHERE id_cart = '.(int) $this->context->cart->id
        );
        $data = json_decode($rawData, true);
        if (!is_array($data)) {
            $data = [];
        }

        $checksum = $this->cartChecksum->generateChecksum($this->context->cart);
        if (isset($data['checksum']) && $data['checksum'] === $checksum) {
            $process->restorePersistedData($data);
        }
    }

}